<?php
session_start();
include('assets/inc/config.php');

// Security headers
header("X-Frame-Options: DENY");
header("X-XSS-Protection: 1; mode=block");
header("X-Content-Type-Options: nosniff");
header("Referrer-Policy: strict-origin-when-cross-origin");
header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline' cdn.ckeditor.com; style-src 'self' 'unsafe-inline'; img-src 'self' data:;");

if(isset($_POST['add_payroll'])) {
    // CSRF protection
    if(!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        die("CSRF token validation failed");
    }

    // Input validation and sanitization
    $pay_number = isset($_POST['pay_number']) ? htmlspecialchars($_POST['pay_number'], ENT_QUOTES, 'UTF-8') : '';
    $pay_doc_name = isset($_POST['pay_doc_name']) ? htmlspecialchars($_POST['pay_doc_name'], ENT_QUOTES, 'UTF-8') : '';
    $pay_doc_number = isset($_POST['pay_doc_number']) ? htmlspecialchars($_POST['pay_doc_number'], ENT_QUOTES, 'UTF-8') : '';
    $pay_doc_email = isset($_POST['pay_doc_email']) ? filter_var($_POST['pay_doc_email'], FILTER_SANITIZE_EMAIL) : '';
    $pay_emp_salary = isset($_POST['pay_emp_salary']) ? filter_var($_POST['pay_emp_salary'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION) : '';
    $pay_descr = isset($_POST['pay_descr']) ? htmlspecialchars($_POST['pay_descr'], ENT_QUOTES, 'UTF-8') : '';
    
    // Validate inputs
    if(empty($pay_number) || empty($pay_doc_name) || empty($pay_doc_number) || empty($pay_doc_email) || empty($pay_emp_salary)) {
        $err = "All required fields must be filled";
    } elseif(!filter_var($pay_doc_email, FILTER_VALIDATE_EMAIL)) {
        $err = "Invalid email format";
    } elseif(!is_numeric($pay_emp_salary) || $pay_emp_salary <= 0) {
        $err = "Salary must be a positive numeric value";
    } else {
        // Prepared statement with error handling
        try {
            $query = "INSERT INTO his_payrolls (pay_number, pay_doc_name, pay_doc_number, pay_doc_email, pay_emp_salary, pay_descr) VALUES(?,?,?,?,?,?)";
            $stmt = $mysqli->prepare($query);
            
            if(!$stmt) {
                throw new Exception("Database error: " . $mysqli->error);
            }
            
            $rc = $stmt->bind_param('ssssss', $pay_number, $pay_doc_name, $pay_doc_number, $pay_doc_email, $pay_emp_salary, $pay_descr);
            
            if(!$stmt->execute()) {
                throw new Exception("Execution failed: " . $stmt->error);
            }
            
            if($stmt->affected_rows > 0) {
                $success = "Payroll Record Added Successfully";
                // Regenerate CSRF token after successful submission
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
                // Clear form fields
                $_POST = array();
            } else {
                $err = "No rows affected. Please try again.";
            }
            
            $stmt->close();
        } catch(Exception $e) {
            $err = "Database error: " . $e->getMessage();
            error_log($e->getMessage());
        }
    }
}

// Generate CSRF token if not exists
if(!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <?php include('assets/inc/head.php');?>
    <meta http-equiv="Content-Security-Policy" content="default-src 'self'; script-src 'self' 'unsafe-inline' cdn.ckeditor.com; style-src 'self' 'unsafe-inline'; img-src 'self' data:;">
    <style>
        .card {
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        .form-control:focus {
            border-color: #6658dd;
            box-shadow: 0 0 0 0.2rem rgba(102, 88, 221, 0.25);
        }
        .page-title-box {
            padding-bottom: 20px;
        }
        .header-title {
            color: #6658dd;
            margin-bottom: 20px;
        }
        .form-group label {
            font-weight: 600;
            color: #495057;
        }
        .btn-primary {
            background-color: #6658dd;
            border-color: #6658dd;
            padding: 10px 25px;
            font-weight: 600;
        }
        .btn-primary:hover {
            background-color: #5749d2;
            border-color: #5749d2;
        }
        .required-field::after {
            content: " *";
            color: red;
        }
        .salary-input {
            position: relative;
        }
        .salary-input::before {
            content: "$";
            position: absolute;
            left: 10px;
            top: 38px;
            font-weight: 600;
            color: #495057;
        }
        .salary-input input {
            padding-left: 25px;
        }
        .alert {
            border-radius: 8px;
        }
        #editor-container {
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 10px;
            min-height: 200px;
        }
    </style>
</head>

<body>
    <div id="wrapper">
        <?php include("assets/inc/nav.php");?>
        <?php include("assets/inc/sidebar.php");?>

        <?php
            $doc_number = $_GET['doc_number'] ?? '';
            if(empty($doc_number)) {
                die("Invalid employee number");
            }
            
            // Secure database query with prepared statement
            $ret = "SELECT * FROM his_docs WHERE doc_number=?";
            $stmt = $mysqli->prepare($ret);
            $stmt->bind_param('s', $doc_number);
            $stmt->execute();
            $res = $stmt->get_result();
            
            if($res->num_rows === 0) {
                die("Employee not found");
            }
            
            $row = $res->fetch_object();
        ?>
        <div class="content-page">
            <div class="content">
                <div class="container-fluid">
                    <div class="row">
                        <div class="col-12">
                            <div class="page-title-box">
                                <div class="page-title-right">
                                    <ol class="breadcrumb m-0">
                                        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                                        <li class="breadcrumb-item"><a href="javascript:void(0);">Payrolls</a></li>
                                        <li class="breadcrumb-item active">Add Payroll Record</li>
                                    </ol>
                                </div>
                                <h4 class="page-title">Add Employee Payroll Record</h4>
                            </div>
                        </div>
                    </div>

                    <?php if(isset($success)): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                                <span aria-hidden="true">&times;</span>
                            </button>
                            <?php echo htmlspecialchars($success); ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if(isset($err)): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                                <span aria-hidden="true">&times;</span>
                            </button>
                            <?php echo htmlspecialchars($err); ?>
                        </div>
                    <?php endif; ?>

                    <div class="row">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-body">
                                    <h4 class="header-title">Employee Payroll Information</h4>
                                    <p class="text-muted">Fill all required fields</p>
                                    
                                    <form method="post" id="payrollForm">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        
                                        <div class="form-row">
                                            <div class="form-group col-md-4">
                                                <label for="pay_doc_name" class="col-form-label required-field">Employee Name</label>
                                                <input type="text" required name="pay_doc_name" value="<?php echo htmlspecialchars($row->doc_fname . ' ' . $row->doc_lname); ?>" class="form-control" id="pay_doc_name" readonly>
                                            </div>

                                            <div class="form-group col-md-4">
                                                <label for="pay_doc_email" class="col-form-label required-field">Employee Email</label>
                                                <input required type="email" name="pay_doc_email" value="<?php echo htmlspecialchars($row->doc_email); ?>" class="form-control" id="pay_doc_email" readonly>
                                            </div>

                                            <div class="form-group col-md-4">
                                                <label for="pay_doc_number" class="col-form-label required-field">Employee Number</label>
                                                <input required type="text" name="pay_doc_number" value="<?php echo htmlspecialchars($row->doc_number); ?>" class="form-control" id="pay_doc_number" readonly>
                                            </div>
                                        </div>

                                        <div class="form-row">
                                            <div class="form-group col-md-6 salary-input">
                                                <label for="pay_emp_salary" class="col-form-label required-field">Employee Salary</label>
                                                <input type="text" required name="pay_emp_salary" class="form-control" id="pay_emp_salary" placeholder="Enter salary amount" value="<?php echo isset($_POST['pay_emp_salary']) ? htmlspecialchars($_POST['pay_emp_salary']) : ''; ?>">
                                                <small class="form-text text-muted">Enter numeric value only</small>
                                            </div>
                                        </div>

                                        <div class="form-group" style="display:none">
                                            <?php 
                                                $length = 5;    
                                                $pay_no = substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 1, $length);
                                            ?>
                                            <input type="text" name="pay_number" value="<?php echo htmlspecialchars($pay_no); ?>" class="form-control">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="editor" class="col-form-label required-field">Payroll Description</label>
                                            <div id="editor-container"><?php echo isset($_POST['pay_descr']) ? htmlspecialchars($_POST['pay_descr']) : ''; ?></div>
                                            <textarea id="pay_descr" name="pay_descr" style="display:none"></textarea>
                                        </div>

                                        <button type="submit" name="add_payroll" class="ladda-button btn btn-primary" data-style="expand-right">
                                            <span class="ladda-label">Add Payroll Record</span>
                                        </button>
                                        <button type="reset" class="btn btn-secondary">Reset Form</button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <?php include('assets/inc/footer.php');?>
        </div>

        <div class="rightbar-overlay"></div>
        
        <!-- Using CKEditor 5 (latest secure version) -->
        <script src="https://cdn.ckeditor.com/ckeditor5/41.3.1/classic/ckeditor.js"></script>
        <script>
            // Initialize CKEditor 5
            ClassicEditor
                .create(document.querySelector('#editor-container'), {
                    toolbar: [
                        'heading', '|', 'bold', 'italic', 'link', 'bulletedList', 
                        'numberedList', 'blockQuote', 'undo', 'redo'
                    ],
                    heading: {
                        options: [
                            { model: 'paragraph', title: 'Paragraph', class: 'ck-heading_paragraph' },
                            { model: 'heading1', view: 'h1', title: 'Heading 1', class: 'ck-heading_heading1' },
                            { model: 'heading2', view: 'h2', title: 'Heading 2', class: 'ck-heading_heading2' }
                        ]
                    }
                })
                .then(editor => {
                    editor.model.document.on('change:data', () => {
                        document.getElementById('pay_descr').value = editor.getData();
                    });
                })
                .catch(error => {
                    console.error(error);
                });

            // Validate salary input
            document.getElementById('pay_emp_salary').addEventListener('input', function(e) {
                this.value = this.value.replace(/[^0-9.]/g, '').replace(/(\..*)\./g, '$1');
            });
            
            // Form validation
            document.getElementById('payrollForm').addEventListener('submit', function(e) {
                let salary = document.getElementById('pay_emp_salary').value;
                if(isNaN(salary) || salary <= 0) {
                    alert('Please enter a valid positive numeric value for salary');
                    e.preventDefault();
                }
                
                // Ensure CKEditor content is properly submitted
                if(document.getElementById('pay_descr').value.trim() === '') {
                    alert('Please enter a payroll description');
                    e.preventDefault();
                }
            });
        </script>

        <script src="assets/js/vendor.min.js"></script>
        <script src="assets/js/app.min.js"></script>
        <script src="assets/libs/ladda/spin.js"></script>
        <script src="assets/libs/ladda/ladda.js"></script>
        <script src="assets/js/pages/loading-btn.init.js"></script>
    </body>
</html>