<?php
header('Content-Type: text/html');
require_once './constant/connect.php';

$order_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($order_id <= 0) {
    die('Invalid order ID');
}

try {
    // Fetch system settings for pharmacy details
    $settingsQuery = "SELECT system_name, phone, email, company FROM system_settings WHERE id = 1";
    $settingsStmt = $connect->prepare($settingsQuery);
    $settingsStmt->execute();
    $settings = $settingsStmt->get_result()->fetch_assoc();

    // Fetch order details with all necessary fields
    $orderQuery = "SELECT o.uno, o.orderDate,  o.clientName, o.clientContact, o.subTotal, o.totalAmount, 
                          o.grandTotalValue, o.paid, o.dueValue, o.discount, o.gstn, o.paymentType, o.paymentStatus,
                          o.type
                   FROM orders o 
                   WHERE o.id = ?";
    $orderStmt = $connect->prepare($orderQuery);
    $orderStmt->bind_param("i", $order_id);
    $orderStmt->execute();
    $orderResult = $orderStmt->get_result();
    $order = $orderResult->fetch_assoc();

    if (!$order) {
        die('Order not found');
    }

    // Fetch order items with all product details including pack, expiry, and real medicine name
    $itemQuery = "SELECT oi.productName, oi.quantity, oi.rate, oi.total, oi.discount_percentage,
                         oi.pack, oi.expiry_date, oi.real_medicine_name,
                         p.bno AS batch_no, p.mrp, p.chamber, p.unit_description,
                         c.categories_name
                  FROM order_item oi
                  LEFT JOIN product p ON oi.product_id = p.product_id
                  LEFT JOIN categories c ON p.categories_id = c.categories_id
                  WHERE oi.order_id = ?";
    $itemStmt = $connect->prepare($itemQuery);
    $itemStmt->bind_param("i", $order_id);
    $itemStmt->execute();
    $itemResult = $itemStmt->get_result();
    $items = $itemResult->fetch_all(MYSQLI_ASSOC);

    // Calculate amounts
    $discount_amount = $order['subTotal'] * ($order['discount'] / 100);
    $taxable_amount = $order['subTotal'] - $discount_amount;
    
    // Calculate tax amounts if needed
    $sgst_percentage = 0;
    $cgst_percentage = 0;
    $sgst_amount = 0;
    $cgst_amount = 0;
    
    // If GSTN is provided, calculate GST (typical 9% CGST + 9% SGST = 18% total)
    if (!empty($order['gstn'])) {
        $sgst_percentage = 9;
        $cgst_percentage = 9;
        $sgst_amount = ($taxable_amount * $sgst_percentage) / 100;
        $cgst_amount = ($taxable_amount * $cgst_percentage) / 100;
    }

    // Payment type mapping
    $paymentTypes = [
        1 => 'Netbanking',
        2 => 'Cash',
        3 => 'Insurance', 
        4 => 'UPI',
        5 => 'Debit Card',
        6 => 'Credit Card'
    ];

    // Payment status mapping
    $paymentStatuses = [
        1 => 'Paid',
        2 => 'Pending'
    ];

} catch (Exception $e) {
    die('Error fetching data: ' . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice - <?php echo htmlspecialchars($order['uno']); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f5f6fa;
            color: #333;
            font-size: 14px;
        }
        .invoice {
            max-width: 900px;
            margin: 0 auto;
            background: #fff;
            padding: 25px;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        .header {
            text-align: center;
            border-bottom: 3px solid #2c3e50;
            padding-bottom: 15px;
            margin-bottom: 20px;
        }
        .header h1 {
            margin: 0;
            color: #2c3e50;
            font-size: 24px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .header p {
            margin: 5px 0;
            font-size: 13px;
            color: #7f8c8d;
        }
        .company-info {
            text-align: center;
            margin-bottom: 20px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 5px;
        }
        .company-info h2 {
            margin: 0 0 8px 0;
            color: #2c3e50;
            font-size: 18px;
        }
        .company-info p {
            margin: 3px 0;
            font-size: 13px;
        }
        .details-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }
        .detail-section {
            font-size: 13px;
            line-height: 1.5;
        }
        .detail-section h3 {
            margin: 0 0 8px 0;
            color: #2c3e50;
            font-size: 14px;
            border-bottom: 1px solid #e0e0e0;
            padding-bottom: 5px;
        }
        .detail-section p {
            margin: 4px 0;
        }
        .table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 12px;
        }
        .table th {
            background-color: #2c3e50;
            color: #fff;
            padding: 10px 8px;
            text-align: left;
            font-weight: 600;
            border: 1px solid #34495e;
        }
        .table td {
            padding: 8px;
            border: 1px solid #e0e0e0;
            vertical-align: top;
        }
        .table tbody tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        .table tbody tr:hover {
            background-color: #e8f4f8;
        }
        .amount-section {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-top: 20px;
        }
        .payment-details, .amount-breakup {
            font-size: 13px;
        }
        .payment-details h3, .amount-breakup h3 {
            margin: 0 0 10px 0;
            color: #2c3e50;
            font-size: 14px;
            border-bottom: 1px solid #e0e0e0;
            padding-bottom: 5px;
        }
        .amount-row {
            display: flex;
            justify-content: space-between;
            margin: 6px 0;
            padding: 3px 0;
        }
        .amount-row.discount {
            color: #e74c3c;
            font-weight: 600;
        }
        .amount-row.total {
            border-top: 1px solid #e0e0e0;
            font-weight: 600;
            color: #2c3e50;
            margin-top: 8px;
            padding-top: 8px;
        }
        .amount-row.grand-total {
            border-top: 2px solid #2c3e50;
            font-weight: 700;
            color: #e74c3c;
            font-size: 15px;
            margin-top: 10px;
            padding-top: 10px;
        }
        .footer {
            text-align: center;
            margin-top: 30px;
            font-size: 12px;
            color: #7f8c8d;
            border-top: 1px dashed #e0e0e0;
            padding-top: 15px;
            line-height: 1.5;
        }
        .signature-section {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px dashed #e0e0e0;
        }
        .signature {
            text-align: center;
        }
        .signature-line {
            border-top: 1px solid #333;
            width: 200px;
            margin: 30px auto 5px auto;
        }
        .barcode-area {
            text-align: center;
            margin: 20px 0;
            padding: 15px;
            border: 1px dashed #ccc;
            background: #f9f9f9;
        }
        @media print {
            body {
                background-color: #fff;
                padding: 0;
                margin: 0;
            }
            .invoice {
                box-shadow: none;
                border: none;
                width: 100%;
                padding: 15px;
                margin: 0;
            }
            .header {
                border-bottom-width: 2px;
            }
            .no-print {
                display: none;
            }
        }
        .invoice-meta {
            display: flex;
            justify-content: space-between;
            margin-bottom: 15px;
            font-size: 13px;
            color: #7f8c8d;
        }
        .highlight {
            background-color: #fff3cd !important;
            font-weight: 600;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <div class="invoice">
        <!-- Company Information -->
        <div class="company-info">
            <h2><?php echo htmlspecialchars($settings['system_name']); ?></h2>
            <p><?php echo htmlspecialchars($settings['company'] ?? 'Pharmacy Business'); ?></p>
            <p><?php echo htmlspecialchars($settings['address'] ?? 'Address not specified'); ?></p>
            <p>Phone: <?php echo htmlspecialchars($settings['phone']); ?> | Email: <?php echo htmlspecialchars($settings['email']); ?></p>
            <?php if (!empty($order['gstn'])): ?>
                <p>GSTIN: <?php echo htmlspecialchars($order['gstn']); ?></p>
            <?php endif; ?>
        </div>

        <!-- Invoice Header -->
        <div class="header">
            <h1>TAX INVOICE</h1>
            <p>Invoice No: <?php echo htmlspecialchars($order['uno']); ?> | Date: <?php echo htmlspecialchars($order['orderDate']); ?> | Time: <?php echo htmlspecialchars($order['orderTime']); ?></p>
            <p>Printed: <?php echo date('d-m-Y H:i A'); ?> | Order Type: <?php echo htmlspecialchars(ucfirst($order['type'])); ?></p>
        </div>

        <!-- Customer and Invoice Details -->
        <div class="details-grid">
            <div class="detail-section">
                <h3>Bill To:</h3>
                <p><strong>Name:</strong> <?php echo htmlspecialchars($order['clientName']); ?></p>
                <p><strong>Contact:</strong> <?php echo htmlspecialchars($order['clientContact']); ?></p>
                <?php if (!empty($order['address'])): ?>
                    <p><strong>Address:</strong> <?php echo htmlspecialchars($order['address']); ?></p>
                <?php endif; ?>
                <?php if (!empty($order['gstn'])): ?>
                    <p><strong>GSTIN:</strong> <?php echo htmlspecialchars($order['gstn']); ?></p>
                <?php endif; ?>
            </div>
            <div class="detail-section">
                <h3>Invoice Details:</h3>
                <p><strong>Invoice No:</strong> <?php echo htmlspecialchars($order['uno']); ?></p>
                <p><strong>Date:</strong> <?php echo htmlspecialchars($order['orderDate']); ?></p>
                <p><strong>Time:</strong> <?php echo htmlspecialchars($order['orderTime']); ?></p>
                <p><strong>Type:</strong> <?php echo htmlspecialchars(ucfirst($order['type'])); ?></p>
            </div>
        </div>

        <!-- Products Table -->
        <table class="table">
            <thead>
                <tr>
                    <th width="5%">#</th>
                    <th width="25%">Product Name</th>
                    <th width="8%">Pack</th>
                    <th width="10%">Batch No</th>
                    <th width="10%">Expiry Date</th>
                    <th width="8%">MRP</th>
                    <th width="8%">Quantity</th>
                    <th width="8%">Rate</th>
                    <th width="8%">Disc %</th>
                    <th width="10%">Amount</th>
                </tr>
            </thead>
            <tbody>
                <?php 
                $counter = 1;
                foreach ($items as $item): 
                    $displayName = !empty($item['real_medicine_name']) ? $item['real_medicine_name'] : $item['productName'];
                ?>
                    <tr>
                        <td class="text-center"><?php echo $counter++; ?></td>
                        <td>
                            <strong><?php echo htmlspecialchars($displayName); ?></strong>
                            <?php if (!empty($item['categories_name'])): ?>
                                <br><small style="color: #666;"><?php echo htmlspecialchars($item['categories_name']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td class="text-center"><?php echo htmlspecialchars($item['pack'] ?? '1*1'); ?></td>
                        <td class="text-center"><?php echo htmlspecialchars($item['batch_no'] ?? 'N/A'); ?></td>
                        <td class="text-center">
                            <?php 
                            if (!empty($item['expiry_date'])) {
                                echo htmlspecialchars(date('m/Y', strtotime($item['expiry_date'])));
                            } else {
                                echo 'N/A';
                            }
                            ?>
                        </td>
                        <td class="text-right"><?php echo number_format($item['mrp'] ?? 0, 2); ?></td>
                        <td class="text-center"><?php echo htmlspecialchars($item['quantity']); ?></td>
                        <td class="text-right"><?php echo number_format($item['rate'], 2); ?></td>
                        <td class="text-center"><?php echo number_format($item['discount_percentage'], 1); ?>%</td>
                        <td class="text-right"><?php echo number_format($item['total'], 2); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <!-- Amount and Payment Details -->
        <div class="amount-section">
            <div class="payment-details">
                <h3>Payment Information</h3>
                <div class="amount-row">
                    <span>Payment Type:</span>
                    <span><?php echo htmlspecialchars($paymentTypes[$order['paymentType']] ?? 'Unknown'); ?></span>
                </div>
                <div class="amount-row">
                    <span>Payment Status:</span>
                    <span><?php echo htmlspecialchars($paymentStatuses[$order['paymentStatus']] ?? 'Unknown'); ?></span>
                </div>
                <div class="amount-row">
                    <span>Order Type:</span>
                    <span><?php echo htmlspecialchars(ucfirst($order['type'])); ?></span>
                </div>
                <?php if (!empty($order['gstn'])): ?>
                    <div class="amount-row">
                        <span>GSTIN:</span>
                        <span><?php echo htmlspecialchars($order['gstn']); ?></span>
                    </div>
                <?php endif; ?>
            </div>

            <div class="amount-breakup">
                <h3>Amount Breakdown</h3>
                <div class="amount-row">
                    <span>Sub Total:</span>
                    <span>₹<?php echo number_format($order['subTotal'], 2); ?></span>
                </div>
                
                <?php if ($order['discount'] > 0): ?>
                    <div class="amount-row discount">
                        <span>Discount (<?php echo number_format($order['discount'], 1); ?>%):</span>
                        <span>-₹<?php echo number_format($discount_amount, 2); ?></span>
                    </div>
                    
                    <div class="amount-row">
                        <span>Amount After Discount:</span>
                        <span>₹<?php echo number_format($taxable_amount, 2); ?></span>
                    </div>
                <?php endif; ?>

                <?php if (!empty($order['gstn'])): ?>
                    <div class="amount-row">
                        <span>CGST (<?php echo $cgst_percentage; ?>%):</span>
                        <span>₹<?php echo number_format($cgst_amount, 2); ?></span>
                    </div>
                    <div class="amount-row">
                        <span>SGST (<?php echo $sgst_percentage; ?>%):</span>
                        <span>₹<?php echo number_format($sgst_amount, 2); ?></span>
                    </div>
                <?php endif; ?>

                <div class="amount-row total">
                    <span>Total Amount:</span>
                    <span>₹<?php echo number_format($order['totalAmount'], 2); ?></span>
                </div>
                 <div class="amount-row total">
                    <span>Discount:</span>
                    <span> ₹<?php echo number_format($discount_amount, 2); ?></span>
                </div>

                <div class="amount-row grand-total">
                    <span>Grand Total:</span>
                    <span>₹<?php echo number_format($order['grandTotalValue'], 2); ?></span>
                </div>

                <div class="amount-row">
                    <span>Paid Amount:</span>
                    <span>₹<?php echo number_format($order['paid'], 2); ?></span>
                </div>

                <div class="amount-row highlight">
                    <span>Due Amount:</span>
                    <span>₹<?php echo number_format($order['dueValue'], 2); ?></span>
                </div>
            </div>
        </div>

      

        <!-- Signature Section -->
        <div class="signature-section">
            <div class="signature">
                <p>Customer Signature</p>
                <div class="signature-line"></div>
            </div>
            <div class="signature">
                <p>Authorized Signature</p>
                <div class="signature-line"></div>
                <p><?php echo htmlspecialchars($settings['system_name']); ?></p>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer">
            <p><strong>Thank you for your business!</strong></p>
            <p>This is a computer generated invoice and does not require a physical signature.</p>
            <p>Powered by <?php echo htmlspecialchars($settings['system_name']); ?> | Printed on: <?php echo date('d-m-Y H:i A'); ?></p>
            <p>For any queries, contact: <?php echo htmlspecialchars($settings['phone']); ?> | <?php echo htmlspecialchars($settings['email']); ?></p>
        </div>

        <!-- Barcode Area (Optional) -->
        <div class="barcode-area no-print">
            <p><strong>Invoice Barcode Area</strong></p>
            <p style="font-family: monospace; font-size: 16px; letter-spacing: 3px;">* <?php echo htmlspecialchars($order['uno']); ?> *</p>
            <p><small>Scan this code for digital verification</small></p>
        </div>
    </div>

    <script>
        // Auto-print after 1 second delay
        setTimeout(function() {
            window.print();
        }, 1000);

        // Add keyboard shortcut to close after printing (Ctrl + W)
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey && e.key === 'w') {
                window.close();
            }
        });
    </script>
</body>
</html>
<?php
$connect->close();
?>